const express = require('express');
const pool = require('../config/database');
const { authenticateToken } = require('../middleware/auth');
const router = express.Router();

// Listar colaboradores
router.get('/', authenticateToken, async (req, res) => {
    try {
        let query = `
            SELECT e.*, u.username as leader_name 
            FROM employees e 
            LEFT JOIN users u ON e.leader_id = u.id
        `;
        let params = [];

        // Se não for Admin, mostrar apenas colaboradores do próprio líder
        if (req.user.role !== 'Admin') {
            query += ' WHERE e.leader_id = ?';
            params.push(req.user.id);
        }

        query += ' ORDER BY e.created_at DESC';

        const [rows] = await pool.execute(query, params);
        res.json(rows);
    } catch (error) {
        console.error('Erro ao buscar colaboradores:', error);
        res.status(500).json({ error: 'Erro interno do servidor' });
    }
});

// Criar novo colaborador
router.post('/', authenticateToken, async (req, res) => {
    try {
        const { name, employee_function } = req.body;

        if (!name) {
            return res.status(400).json({ error: 'Nome é obrigatório' });
        }

        const [result] = await pool.execute(
            'INSERT INTO employees (name, employee_function, leader_id) VALUES (?, ?, ?)',
            [name, employee_function, req.user.id]
        );

        // Log de auditoria
        await pool.execute(
            'INSERT INTO audit_logs (user_id, action, details) VALUES (?, ?, ?)',
            [req.user.id, 'CREATE_EMPLOYEE', `Criou colaborador ${name} com função ${employee_function}`]
        );

        res.status(201).json({
            id: result.insertId,
            name,
            employee_function,
            leader_id: req.user.id,
            message: 'Colaborador criado com sucesso'
        });
    } catch (error) {
        console.error('Erro ao criar colaborador:', error);
        res.status(500).json({ error: 'Erro interno do servidor' });
    }
});

// Atualizar colaborador
router.put('/:id', authenticateToken, async (req, res) => {
    try {
        const { id } = req.params;
        const { name, employee_function } = req.body;

        if (!name) {
            return res.status(400).json({ error: 'Nome é obrigatório' });
        }

        // Verificar se o colaborador pertence ao usuário (se não for Admin)
        let query = 'UPDATE employees SET name = ?, employee_function = ? WHERE id = ?';
        let params = [name, employee_function, id];

        if (req.user.role !== 'Admin') {
            query += ' AND leader_id = ?';
            params.push(req.user.id);
        }

        const [result] = await pool.execute(query, params);

        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Colaborador não encontrado ou sem permissão' });
        }

        // Log de auditoria
        await pool.execute(
            'INSERT INTO audit_logs (user_id, action, details) VALUES (?, ?, ?)',
            [req.user.id, 'UPDATE_EMPLOYEE', `Atualizou colaborador ID ${id}`]
        );

        res.json({ message: 'Colaborador atualizado com sucesso' });
    } catch (error) {
        console.error('Erro ao atualizar colaborador:', error);
        res.status(500).json({ error: 'Erro interno do servidor' });
    }
});

// Deletar colaborador
router.delete('/:id', authenticateToken, async (req, res) => {
    try {
        const { id } = req.params;

        // Verificar se o colaborador existe e pertence ao usuário (se não for Admin)
        let selectQuery = 'SELECT name FROM employees WHERE id = ?';
        let selectParams = [id];

        if (req.user.role !== 'Admin') {
            selectQuery += ' AND leader_id = ?';
            selectParams.push(req.user.id);
        }

        const [employee] = await pool.execute(selectQuery, selectParams);

        if (employee.length === 0) {
            return res.status(404).json({ error: 'Colaborador não encontrado ou sem permissão' });
        }

        // Deletar colaborador
        let deleteQuery = 'DELETE FROM employees WHERE id = ?';
        let deleteParams = [id];

        if (req.user.role !== 'Admin') {
            deleteQuery += ' AND leader_id = ?';
            deleteParams.push(req.user.id);
        }

        await pool.execute(deleteQuery, deleteParams);

        // Log de auditoria
        await pool.execute(
            'INSERT INTO audit_logs (user_id, action, details) VALUES (?, ?, ?)',
            [req.user.id, 'DELETE_EMPLOYEE', `Deletou colaborador ${employee[0].name}`]
        );

        res.json({ message: 'Colaborador deletado com sucesso' });
    } catch (error) {
        console.error('Erro ao deletar colaborador:', error);
        res.status(500).json({ error: 'Erro interno do servidor' });
    }
});

module.exports = router;

