const express = require('express');
const pool = require('../config/database');
const { authenticateToken } = require('../middleware/auth');
const router = express.Router();

// Listar flags com filtros
router.get('/', authenticateToken, async (req, res) => {
    try {
        const { status, employee_id, leader_id, start_date, end_date, include_archived } = req.query;

        let query = `
            SELECT f.*, e.name as employee_name, u.username as leader_name 
            FROM flags f 
            LEFT JOIN employees e ON f.employee_id = e.id 
            LEFT JOIN users u ON f.leader_id = u.id
            WHERE 1=1
        `;
        let params = [];

        // Se não for Admin, mostrar apenas flags do próprio líder
        if (req.user.role !== 'Admin') {
            query += ' AND f.leader_id = ?';
            params.push(req.user.id);
        }

        // Por padrão, não mostrar flags arquivadas a menos que especificado
        if (!include_archived || include_archived !== 'true') {
            query += ' AND f.status IN ("Aberta", "Em análise", "Em plano de ação")';
        }

        // Filtros
        if (status) {
            query += ' AND f.status = ?';
            params.push(status);
        }

        if (employee_id) {
            query += ' AND f.employee_id = ?';
            params.push(employee_id);
        }

        if (leader_id) {
            query += ' AND f.leader_id = ?';
            params.push(leader_id);
        }

        if (start_date) {
            query += ' AND DATE(f.occurred_at) >= ?';
            params.push(start_date);
        }

        if (end_date) {
            query += ' AND DATE(f.occurred_at) <= ?';
            params.push(end_date);
        }

        query += ' ORDER BY f.created_at DESC';

        const [rows] = await pool.execute(query, params);
        res.json(rows);
    } catch (error) {
        console.error('Erro ao buscar flags:', error);
        res.status(500).json({ error: 'Erro interno do servidor' });
    }
});

// Criar nova flag
router.post("/", authenticateToken, async (req, res) => {
    console.log("Request body for flag creation:", req.body);
    try {
        const { 
            title, 
            description, 
            employee_id, 
            category, 
            occurred_at, 
            gravity, 
            impact_area, 
            impact_other,
            status = 'Aberta'
        } = req.body;

        if (!title || !employee_id || !category || !occurred_at || !gravity) {
            return res.status(400).json({ 
                error: 'Título, colaborador, categoria, data do ocorrido e gravidade são obrigatórios' 
            });
        }

        // Validar valores de gravidade
        const validGravities = ['Amarela', 'Laranja', 'Vermelha'];
        if (!validGravities.includes(gravity)) {
            return res.status(400).json({ error: 'Gravidade inválida' });
        }

        // Verificar se o colaborador pertence ao usuário (se não for Admin)
        if (req.user.role !== 'Admin') {
            const [employee] = await pool.execute(
                'SELECT id FROM employees WHERE id = ? AND leader_id = ?',
                [employee_id, req.user.id]
            );

            if (employee.length === 0) {
                return res.status(400).json({ error: 'Colaborador não encontrado ou sem permissão' });
            }
        }

        const [result] = await pool.execute(
            `INSERT INTO flags (
                title, description, employee_id, leader_id, category, 
                occurred_at, gravity, impact_area, impact_other, status
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
            [
                title, description, employee_id, req.user.id, category,
                occurred_at, gravity, impact_area, impact_other, status
            ]
        );

        // Log de auditoria
        await pool.execute(
            'INSERT INTO audit_logs (user_id, action, details) VALUES (?, ?, ?)',
            [req.user.id, 'CREATE_FLAG', `Criou flag "${title}" para colaborador ID ${employee_id}`]
        );

        res.status(201).json({
            id: result.insertId,
            title,
            description,
            employee_id,
            leader_id: req.user.id,
            category,
            occurred_at,
            gravity,
            impact_area,
            impact_other,
            status,
            message: 'Flag criada com sucesso'
        });
    } catch (error) {
        console.error('Erro detalhado ao criar flag:', error.message, error.stack);
        res.status(500).json({ error: 'Erro interno do servidor ao criar flag', details: error.message });
    }
});

// Atualizar flag
router.put('/:id', authenticateToken, async (req, res) => {
    try {
        const { id } = req.params;
        const { 
            title, 
            description, 
            employee_id, 
            category, 
            occurred_at, 
            gravity, 
            impact_area, 
            impact_other,
            status
        } = req.body;

        if (!title) {
            return res.status(400).json({ error: 'Título é obrigatório' });
        }

        // Verificar se a flag pertence ao usuário (se não for Admin)
        let checkQuery = 'SELECT * FROM flags WHERE id = ?';
        let checkParams = [id];

        if (req.user.role !== 'Admin') {
            checkQuery += ' AND leader_id = ?';
            checkParams.push(req.user.id);
        }

        const [existingFlag] = await pool.execute(checkQuery, checkParams);

        if (existingFlag.length === 0) {
            return res.status(404).json({ error: 'Flag não encontrada ou sem permissão' });
        }

        // Verificar se o colaborador pertence ao usuário (se não for Admin e employee_id foi fornecido)
        if (employee_id && req.user.role !== 'Admin') {
            const [employee] = await pool.execute(
                'SELECT id FROM employees WHERE id = ? AND leader_id = ?',
                [employee_id, req.user.id]
            );

            if (employee.length === 0) {
                return res.status(400).json({ error: 'Colaborador não encontrado ou sem permissão' });
            }
        }

        // Construir query de atualização
        let updateFields = ['title = ?'];
        let updateParams = [title];

        if (description !== undefined) {
            updateFields.push('description = ?');
            updateParams.push(description);
        }

        if (employee_id !== undefined) {
            updateFields.push('employee_id = ?');
            updateParams.push(employee_id);
        }

        if (category !== undefined) {
            updateFields.push('category = ?');
            updateParams.push(category);
        }

        if (occurred_at !== undefined) {
            updateFields.push('occurred_at = ?');
            updateParams.push(occurred_at);
        }

        if (gravity !== undefined) {
            updateFields.push('gravity = ?');
            updateParams.push(gravity);
        }

        if (impact_area !== undefined) {
            updateFields.push('impact_area = ?');
            updateParams.push(impact_area);
        }

        if (impact_other !== undefined) {
            updateFields.push('impact_other = ?');
            updateParams.push(impact_other);
        }

        if (status !== undefined) {
            updateFields.push('status = ?');
            updateParams.push(status);
        }

        let query = `UPDATE flags SET ${updateFields.join(', ')} WHERE id = ?`;
        updateParams.push(id);

        if (req.user.role !== 'Admin') {
            query += ' AND leader_id = ?';
            updateParams.push(req.user.id);
        }

        await pool.execute(query, updateParams);

        // Log de auditoria
        await pool.execute(
            'INSERT INTO audit_logs (user_id, action, details) VALUES (?, ?, ?)',
            [req.user.id, 'UPDATE_FLAG', `Atualizou flag ID ${id}`]
        );

        res.json({ message: 'Flag atualizada com sucesso' });
    } catch (error) {
        console.error('Erro ao atualizar flag:', error);
        res.status(500).json({ error: 'Erro interno do servidor' });
    }
});

// Atualizar apenas o status da flag (para edição rápida no dashboard)
router.patch('/:id/status', authenticateToken, async (req, res) => {
    try {
        const { id } = req.params;
        const { status } = req.body;

        if (!status) {
            return res.status(400).json({ error: 'Status é obrigatório' });
        }

        const validStatuses = ['Aberta', 'Em análise', 'Em plano de ação', 'Concluída', 'Arquivada'];
        if (!validStatuses.includes(status)) {
            return res.status(400).json({ error: 'Status inválido' });
        }

        // Verificar se a flag pertence ao usuário (se não for Admin)
        let query = 'UPDATE flags SET status = ? WHERE id = ?';
        let params = [status, id];

        if (req.user.role !== 'Admin') {
            query += ' AND leader_id = ?';
            params.push(req.user.id);
        }

        const [result] = await pool.execute(query, params);

        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Flag não encontrada ou sem permissão' });
        }

        // Log de auditoria
        await pool.execute(
            'INSERT INTO audit_logs (user_id, action, details) VALUES (?, ?, ?)',
            [req.user.id, 'UPDATE_FLAG_STATUS', `Alterou status da flag ID ${id} para ${status}`]
        );

        res.json({ message: 'Status da flag atualizado com sucesso' });
    } catch (error) {
        console.error('Erro ao atualizar status da flag:', error);
        res.status(500).json({ error: 'Erro interno do servidor' });
    }
});

// Deletar flag
router.delete('/:id', authenticateToken, async (req, res) => {
    try {
        const { id } = req.params;

        // Verificar se a flag existe e pertence ao usuário (se não for Admin)
        let selectQuery = 'SELECT title FROM flags WHERE id = ?';
        let selectParams = [id];

        if (req.user.role !== 'Admin') {
            selectQuery += ' AND leader_id = ?';
            selectParams.push(req.user.id);
        }

        const [flag] = await pool.execute(selectQuery, selectParams);

        if (flag.length === 0) {
            return res.status(404).json({ error: 'Flag não encontrada ou sem permissão' });
        }

        // Deletar flag
        let deleteQuery = 'DELETE FROM flags WHERE id = ?';
        let deleteParams = [id];

        if (req.user.role !== 'Admin') {
            deleteQuery += ' AND leader_id = ?';
            deleteParams.push(req.user.id);
        }

        await pool.execute(deleteQuery, deleteParams);

        // Log de auditoria
        await pool.execute(
            'INSERT INTO audit_logs (user_id, action, details) VALUES (?, ?, ?)',
            [req.user.id, 'DELETE_FLAG', `Deletou flag "${flag[0].title}"`]
        );

        res.json({ message: 'Flag deletada com sucesso' });
    } catch (error) {
        console.error('Erro ao deletar flag:', error);
        res.status(500).json({ error: 'Erro interno do servidor' });
    }
});

module.exports = router;

