const express = require('express');
const pool = require('../config/database');
const { authenticateToken, requireAdmin } = require('../middleware/auth');
const router = express.Router();

// Listar todos os usuários (apenas Admin)
router.get('/', authenticateToken, requireAdmin, async (req, res) => {
    try {
        const [rows] = await pool.execute(
            'SELECT id, username, role, created_at FROM users ORDER BY created_at DESC'
        );
        res.json(rows);
    } catch (error) {
        console.error('Erro ao buscar usuários:', error);
        res.status(500).json({ error: 'Erro interno do servidor' });
    }
});

// Criar novo usuário (apenas Admin)
router.post('/', authenticateToken, requireAdmin, async (req, res) => {
    try {
        const { username, password, role } = req.body;

        if (!username || !password || !role) {
            return res.status(400).json({ error: 'Todos os campos são obrigatórios' });
        }

        if (!['Admin', 'Lider'].includes(role)) {
            return res.status(400).json({ error: 'Role deve ser Admin ou Lider' });
        }

        // Verificar se o username já existe
        const [existing] = await pool.execute(
            'SELECT id FROM users WHERE username = ?',
            [username]
        );

        if (existing.length > 0) {
            return res.status(400).json({ error: 'Username já existe' });
        }

        const [result] = await pool.execute(
            'INSERT INTO users (username, password, role) VALUES (?, ?, ?)',
            [username, password, role]
        );

        // Log de auditoria
        await pool.execute(
            'INSERT INTO audit_logs (user_id, action, details) VALUES (?, ?, ?)',
            [req.user.id, 'CREATE_USER', `Criou usuário ${username} com role ${role}`]
        );

        res.status(201).json({
            id: result.insertId,
            username,
            role,
            message: 'Usuário criado com sucesso'
        });
    } catch (error) {
        console.error('Erro ao criar usuário:', error);
        res.status(500).json({ error: 'Erro interno do servidor' });
    }
});

// Atualizar usuário (apenas Admin)
router.put('/:id', authenticateToken, requireAdmin, async (req, res) => {
    try {
        const { id } = req.params;
        const { username, password, role } = req.body;

        if (!username || !role) {
            return res.status(400).json({ error: 'Username e role são obrigatórios' });
        }

        if (!['Admin', 'Lider'].includes(role)) {
            return res.status(400).json({ error: 'Role deve ser Admin ou Lider' });
        }

        let query = 'UPDATE users SET username = ?, role = ?';
        let params = [username, role];

        if (password) {
            query += ', password = ?';
            params.push(password);
        }

        query += ' WHERE id = ?';
        params.push(id);

        const [result] = await pool.execute(query, params);

        if (result.affectedRows === 0) {
            return res.status(404).json({ error: 'Usuário não encontrado' });
        }

        // Log de auditoria
        await pool.execute(
            'INSERT INTO audit_logs (user_id, action, details) VALUES (?, ?, ?)',
            [req.user.id, 'UPDATE_USER', `Atualizou usuário ID ${id}`]
        );

        res.json({ message: 'Usuário atualizado com sucesso' });
    } catch (error) {
        console.error('Erro ao atualizar usuário:', error);
        res.status(500).json({ error: 'Erro interno do servidor' });
    }
});

// Deletar usuário (apenas Admin)
router.delete('/:id', authenticateToken, requireAdmin, async (req, res) => {
    try {
        const { id } = req.params;

        // Verificar se o usuário existe
        const [user] = await pool.execute(
            'SELECT username FROM users WHERE id = ?',
            [id]
        );

        if (user.length === 0) {
            return res.status(404).json({ error: 'Usuário não encontrado' });
        }

        const [result] = await pool.execute(
            'DELETE FROM users WHERE id = ?',
            [id]
        );

        // Log de auditoria
        await pool.execute(
            'INSERT INTO audit_logs (user_id, action, details) VALUES (?, ?, ?)',
            [req.user.id, 'DELETE_USER', `Deletou usuário ${user[0].username}`]
        );

        res.json({ message: 'Usuário deletado com sucesso' });
    } catch (error) {
        console.error('Erro ao deletar usuário:', error);
        res.status(500).json({ error: 'Erro interno do servidor' });
    }
});

module.exports = router;

